/*
 * @(#)ConceptualERModel.java  1.0  2006-03-20
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.cer.model;

import ch.hslu.cm.simulation.*;
import java.util.*;
import org.jhotdraw.util.ResourceBundleUtil;

/**
 * Represents a Conceptual Entity-Relationship Model.
 *
 * @author Werner Randelshofer
 * @version 1.0 2006-03-20 Created.
 */
public class ConceptualERModel extends AbstractSimulation {
    public static ResourceBundleUtil labels =
            ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels");
    /**
     * Enumeration of simulated concepts.
     */
    public final static int ENTITY_SET = 0;
    public final static int RELATIONSHIP_SET = 1;
    public final static int ISA = 2;
    public final static int RELATIONSHIP = 3;
    public final static int SPECIALIZATION = 4;
    public final static int GENERALIZATION = 5;
    public final static int ATTRIBUTE = 6;
    public final static int ATTRIBUTE_LINK = 7;
    /**
     * A connection can have an undefined concept during construction of a
     * ConceptualERModel.
     */
    public final static int UNDEFINED = -1;
    
    /** Creates a new instance. */
    public ConceptualERModel() {
    }
    /*
    public void setLocale(Locale l) {
        labels = ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels", l);
    }*/
    
    
    public CEREntitySet getSimulatedEntity(String name) {
        // FIXME - Linear search!!
        // This is very slow, we should consider using hash maps for all
        // names
        
        for (Iterator i = getElements().iterator(); i.hasNext(); ) {
            SimulatedElement elem = (SimulatedElement) i.next();
            if (elem.getSimulatedConcept() == ENTITY_SET) {
                CEREntitySet set = (CEREntitySet) elem;
                if (set.getName().equals(name)) {
                    return set;
                }
            }
        }
        
        return null;
    }
    
    public Collection<CEREntitySet> getSimulatedEntitySets() {
        LinkedList<CEREntitySet> result = new LinkedList<CEREntitySet>();
        
        for (SimulatedElement elem : getElements()) {
            if (elem.getSimulatedConcept() == ENTITY_SET) {
                result.add((CEREntitySet) elem);
            }
        }
        
        Collections.sort(result, new CEREntitySetNameComparator());
        
        return result;
    }
    public Collection<CERRelationshipSet> getSimulatedRelationshipSets() {
        LinkedList<CERRelationshipSet> result = new LinkedList<CERRelationshipSet>();
        
        for (SimulatedElement elem : getElements()) {
            if (elem.getSimulatedConcept() == RELATIONSHIP_SET) {
                result.add((CERRelationshipSet) elem);
            }
        }
        
        return result;
    }
    public Collection<CERConnection> getSimulatedRelationships() {
        LinkedList<CERConnection> result = new LinkedList<CERConnection>();
        
        for (SimulatedElement elem : getElements()) {
            if (elem.getSimulatedConcept() == RELATIONSHIP) {
                result.add((CERConnection) elem);
            }
        }
        return result;
    }
    
    public SimulatedObject create(int simulatedConcept) {
        SimulatedObject elem;
        switch (simulatedConcept) {
            case ENTITY_SET :
                elem = new CEREntitySet();
                break;
            case RELATIONSHIP_SET :
                elem = new CERRelationshipSet();
                break;
            case ISA :
                elem = new CERISA();
                break;
            case RELATIONSHIP :
            case GENERALIZATION :
            case SPECIALIZATION :
                elem = new CERConnection();
                break;
            default :
                throw new IllegalArgumentException(simulatedConcept+"");
        }
        return elem;
    }
    
    /**
     * Returns null, if this ConceptualERModel is equivalent to the specified ConceptualERModel.
     * Returns a String describing the difference to the specified ConceptualERModel.
     * Returns null, if the two models are equivalent.
     */
    public String describeDifferencesTo(ConceptualERModel that) {
        // Quickly return null when attempting to compare with ourselves.
        if (that == this) return null;
        
        // This StringBuilder holds the description of the differences
        StringBuilder diff = new StringBuilder();
        
        // We put all elements for which we haven't found a mapping yet
        // into a bag. If we find a mapping, we remove the entity from the bag.
        // The bag is a set, because the same entity can occur only once in a model.
        HashSet<SimulatedElement> thatElementBag = new HashSet<SimulatedElement>(that.getElements());
        HashSet<SimulatedElement> thisElementBag = new HashSet<SimulatedElement>(this.getElements());
        HashSet<SimulatedRelationship> thatConnectionBag = new HashSet<SimulatedRelationship>(that.getRelationships());
        HashSet<SimulatedRelationship> thisConnectionBag = new HashSet<SimulatedRelationship>(this.getRelationships());
        
        // Mappings between this model and that model.
        // For each mapping we found, we put an entry into the map of which
        // the key is in this model, and the value is in that model.
        HashMap<SimulatedElement,SimulatedElement> elementMap = new HashMap<SimulatedElement,SimulatedElement>();
        HashMap<SimulatedRelationship,SimulatedRelationship> connectionMap = new HashMap<SimulatedRelationship,SimulatedRelationship>();
        
        // Find matching elements
        // ----------------------
        for (int quality = 5; quality > 0; quality--) {
            int match;
            do {
                match = 0;
                elementMatcher: for (SimulatedElement thisE : thisElementBag) {
                    for (SimulatedElement thatE : thatElementBag) {
                        match = 0;
                        
                        if (thisE.isEquivalent(thatE)) {
                            match++;
                        }
                        
                        // Match concept. We only do further matchings if the
                        // simulated concepts are the same.
                        if (thisE.getSimulatedConcept() == thatE.getSimulatedConcept()) {
                            match++;
                            
                            String thisN = thisE.getName();
                            String thatN = thatE.getName();
                            if (thisN == null && thatN == null ||
                                    thisN != null && thatN != null && thisN.equals(thatN)) {
                                match++;
                            }
                            
                            HashSet<SimulatedRelationship> thisECBag = new HashSet<SimulatedRelationship>(thisE.getRelationships());
                            HashSet<SimulatedRelationship> thatECBag = new HashSet<SimulatedRelationship>(thatE.getRelationships());
                            if (thisECBag.size() == thatECBag.size()) {
                                match++;
                                
                                HashMap<SimulatedRelationship,SimulatedRelationship> ccMap = new HashMap<SimulatedRelationship,SimulatedRelationship>();
                                boolean matchingPairFound;
                                do {
                                    matchingPairFound = false;
                                    ccMatcher: for (SimulatedRelationship thisEC : thisECBag) {
                                        for (SimulatedRelationship thatEC : thatECBag) {
                                            thisN = thisE.getName();
                                            thatN = thatE.getName();
                                            if ((thisN == null && thatN == null ||
                                                    thisN != null && thatN != null && thisN.equals(thatN)) &&
                                                    thisEC.getSimulatedConcept() == thatEC.getSimulatedConcept() &&
                                                    thisEC.getConnected(thisE).getSimulatedConcept() == thatEC.getConnected(thatE).getSimulatedConcept()
                                                    ) {
                                                thisN = thisEC.getConnected(thisE).getName();
                                                thatN = thatEC.getConnected(thatE).getName();
                                                if (thisN == null && thatN == null ||
                                                        thisN != null && thatN != null && thisN.equals(thatN)) {
                                                    ccMap.put(thisEC, thatEC);
                                                    thisECBag.remove(thisEC);
                                                    thatECBag.remove(thatEC);
                                                    matchingPairFound = true;
                                                    break ccMatcher;
                                                }
                                            }
                                        }
                                    }
                                } while (matchingPairFound);
                                
                                if (thisECBag.size() == 0 && thatECBag.size() == 0) {
                                    match++;
                                }
                            }
                        }
                        if (match == quality) {
                            elementMap.put(thisE, thatE);
                            thisElementBag.remove(thisE);
                            thatElementBag.remove(thatE);
                            break elementMatcher;
                        }
                    }
                }
            } while (match == quality && thisElementBag.size()  > 0 && thatElementBag.size() > 0);
        }
        // Find matching connections
        // ----------------------
        for (int quality = 6; quality > 0; quality--) {
            int match;
            do {
                match = 0;
                connectionMatcher: for (SimulatedRelationship thisC : thisConnectionBag) {
                    for (SimulatedRelationship thatC : thatConnectionBag) {
                        match = 0;
                        
                        if (thisC.isEquivalent(thatC)) {
                            match++;
                        }
                        
                        // Match concept. We only do further matchings if the
                        // simulated concepts are the same.
                        if (thisC.getSimulatedConcept() == thatC.getSimulatedConcept()) {
                            match++;
                            
                            // Match name of connection
                            String thisN = thisC.getName();
                            String thatN = thatC.getName();
                            if (thisN == null && thatN == null ||
                                    thisN != null && thatN != null && thisN.equals(thatN)) {
                                match++;
                            }
                            
                            // Match known connection targets
                            if (elementMap.containsKey(thisC.getStart()) && elementMap.containsKey(thisC.getEnd()) &&
                                    (elementMap.get(thisC.getStart()) == thatC.getStart() &&
                                    elementMap.get(thisC.getEnd()) == thatC.getEnd()) ||
                                    elementMap.get(thisC.getStart()) == thatC.getEnd() &&
                                    elementMap.get(thisC.getEnd()) == thatC.getStart()) {
                                
                                match++;
                            }
                            
                            // Match connection targets by concept
                            if (thisC.getStart().getSimulatedConcept() == thatC.getStart().getSimulatedConcept() &&
                                    thisC.getEnd().getSimulatedConcept() == thatC.getEnd().getSimulatedConcept() ||
                                    thisC.getStart().getSimulatedConcept() == thatC.getEnd().getSimulatedConcept() &&
                                    thisC.getEnd().getSimulatedConcept() == thatC.getStart().getSimulatedConcept()) {
                                match++;
                            }
                            
                            // Match connection targets by name
                            String thisCSName = thisC.getStart().getName();
                            String thatCSName = thatC.getStart().getName();
                            String thisCEName = thisC.getEnd().getName();
                            String thatCEName = thatC.getEnd().getName();
                            if ((thisCSName != null && thatCSName != null &&
                                    thisCEName != null && thatCEName != null) &&
                                    (thisCSName.equals(thatCSName) && thisCEName.equals(thatCEName) ||
                                    thisCSName.equals(thatCEName) && thisCEName.equals(thatCSName))
                                    ) {
                                match++;
                            }
                        }
                        
                        if (match == quality) {
                            connectionMap.put(thisC, thatC);
                            thisConnectionBag.remove(thisC);
                            thatConnectionBag.remove(thatC);
                            break connectionMatcher;
                        }
                    }
                }
            } while (match == quality && thisConnectionBag.size()  > 0 && thatConnectionBag.size() > 0);
        }
        
        // Create an inverse mapping
        HashMap<SimulatedElement,SimulatedElement> inverseElementMap = new HashMap<SimulatedElement,SimulatedElement>();
        for (Map.Entry<SimulatedElement, SimulatedElement> entry : elementMap.entrySet()) {
            inverseElementMap.put(entry.getValue(), entry.getKey());
        }
        HashMap<SimulatedRelationship,SimulatedRelationship> inverseConnectionMap = new HashMap<SimulatedRelationship,SimulatedRelationship>();
        for (Map.Entry<SimulatedRelationship, SimulatedRelationship> entry : connectionMap.entrySet()) {
            inverseConnectionMap.put(entry.getValue(), entry.getKey());
        }
        
        // Report differences in element counts
        /*
        if (this.getElementCount() != that.getElementCount()) {
            diff.append("<li>");
            diff.append(labels.getFormatted("diffElementCount", this.getElementCount(), that.getElementCount()));
            diff.append("</li>");
        }
        // Report differences in connection counts
        if (this.getConnectionCount() != that.getConnectionCount()) {
            diff.append("<li>");
            diff.append(labels.getFormatted("diffConnectionCount", this.getConnectionCount(), that.getConnectionCount()));
            diff.append("</li>");
        }*/
        
        // Report which elements are too much/too few
        for (SimulatedElement e : thisElementBag) {
            if (e.getName() == null) {
                diff.append("<li>");
                diff.append(labels.getFormatted("diffUnnamedElementTooMuch", labels.getFormatted("conceptName", e.getSimulatedConcept())));
                diff.append("</li>");
            } else {
                diff.append("<li>");
                diff.append(labels.getFormatted("diffElementTooMuch", labels.getFormatted("conceptName", e.getSimulatedConcept()), e.getName()));
                diff.append("</li>");
            }
        }
        for (SimulatedElement e : thatElementBag) {
            
            if (e.getName() == null) {
                diff.append("<li>");
                diff.append(labels.getFormatted("diffUnnamedElementMissing", labels.getFormatted("conceptName", e.getSimulatedConcept())));
                diff.append("</li>");
            } else {
                diff.append("<li>");
                diff.append(labels.getFormatted("diffElementMissing", labels.getFormatted("conceptName", e.getSimulatedConcept()), e.getName()));
                diff.append("</li>");
            }
        }
        // Report which connections are too much/too few
        for (SimulatedRelationship c : thisConnectionBag) {
            
            if (elementMap.containsKey(c.getStart()) && elementMap.containsKey(c.getEnd())) {
                SimulatedObject startE = c.getStart();
                SimulatedObject endE = c.getEnd();
                diff.append("<li>");
                diff.append(labels.getFormatted("diffConnectionTooMuch",
                        labels.getFormatted("conceptName", c.getSimulatedConcept()),
                        (startE.getName() == null) ? labels.getFormatted("conceptName", startE.getSimulatedConcept()) : startE.getName(),
                        (endE.getName() == null) ? labels.getFormatted("conceptName", endE.getSimulatedConcept()) : endE.getName()
                        ));
                diff.append("</li>");
            }
        }
        for (SimulatedRelationship c : thatConnectionBag) {
            if (inverseElementMap.containsKey(c.getStart()) && inverseElementMap.containsKey(c.getEnd())) {
                SimulatedObject startE = inverseElementMap.get(c.getStart());
                SimulatedObject endE = inverseElementMap.get(c.getEnd());
                if (startE != null && endE != null) {
                    
                    diff.append("<li>");
                    diff.append(labels.getFormatted("diffConnectionMissing",
                            labels.getFormatted("conceptName", c.getSimulatedConcept()),
                            (startE.getName() == null) ? labels.getFormatted("conceptName", startE.getSimulatedConcept()) : startE.getName(),
                            (endE.getName() == null) ? labels.getFormatted("conceptName", endE.getSimulatedConcept()) : endE.getName()
                            ));
                    diff.append("</li>");
                }
            }
        }
        // Report differences in element properties
        for (Map.Entry<SimulatedElement, SimulatedElement> entry : elementMap.entrySet()) {
            SimulatedObject thisE = entry.getKey();
            SimulatedObject thatE = entry.getValue();
            String thisN = thisE.getName();
            String thatN = thatE.getName();
            if (((thisN == null) != (thatN == null)) ||
                    (thisN != null && thatN != null && ! thisN.equals(thatN))) {
                
                diff.append("<li>");
                diff.append(labels.getFormatted("diffElementName", labels.getFormatted("conceptName", thisE.getSimulatedConcept()), thisE.getName(), thatE.getName()));
                diff.append("</li>");
            }
            if (thisE.getSimulatedConcept() != thatE.getSimulatedConcept()) {
                
                diff.append("<li>");
                diff.append(labels.getFormatted("diffElementConcept", thisE.getSimulatedConcept(), thatE.getSimulatedConcept()));
                diff.append("</li>");
            }
        }
        // Report differences in connection properties
        for (Map.Entry<SimulatedRelationship, SimulatedRelationship> entry : connectionMap.entrySet()) {
            SimulatedRelationship thisE = entry.getKey();
            SimulatedRelationship thatE = entry.getValue();
            
            if (elementMap.containsKey(thisE.getStart()) &&
                    elementMap.containsKey(thisE.getEnd()) &&
                    inverseElementMap.containsKey(thatE.getStart()) &&
                    inverseElementMap.containsKey(thatE.getEnd()) &&
                    
                    ! (elementMap.get(thisE.getStart()) == thatE.getStart() &&
                    elementMap.get(thisE.getEnd()) == thatE.getEnd() ||
                    elementMap.get(thisE.getStart()) == thatE.getEnd() &&
                    elementMap.get(thisE.getEnd()) == thatE.getStart())) {
                
                diff.append("<li>");
                diff.append(labels.getFormatted("diffConnectionEnds",
                        labels.getFormatted("conceptName", thisE.getSimulatedConcept()),
                        thisE.getStart().getName(),
                        thisE.getEnd().getName(),
                        inverseElementMap.get(thatE.getStart()).getName(),
                        inverseElementMap.get(thatE.getEnd()).getName()
                        ));
                diff.append("</li>");
            }
            String thisN = thisE.getName();
            String thatN = thatE.getName();
            if (((thisN == null) != (thatN == null)) ||
                    (thisN != null && thatN != null && ! thisN.equals(thatN))) {
                if ((thatE instanceof CERConnection) && ((CERConnection) thatE).isTraversable()) {
                    diff.append("<li>");
                    diff.append(labels.getFormatted("diffConnectionName", labels.getFormatted("conceptName", thisE.getSimulatedConcept()), thisE.getName(), thatE.getName()));
                    diff.append("</li>");
                }
            }
            if (thisE.getSimulatedConcept() != thatE.getSimulatedConcept()) {
                
                diff.append("<li>");
                diff.append(labels.getFormatted("diffConnectionConcept", thisE.getSimulatedConcept(), thatE.getSimulatedConcept()));
                diff.append("</li>");
            }
        }
        
        // Report ConceptualERModel specific differences
        // ------------------------------------------
        
        // Report differencies between CEREntitySet properties
        for (Map.Entry<SimulatedElement, SimulatedElement> entry : elementMap.entrySet()) {
            if ((entry.getKey() instanceof CEREntitySet) && (entry.getValue() instanceof CEREntitySet)) {
                CEREntitySet thisE = (CEREntitySet) entry.getKey();
                CEREntitySet thatE = (CEREntitySet) entry.getValue();
                if (thisE.getType() != thatE.getType()) {
                    
                    diff.append("<li>");
                    diff.append(labels.getFormatted("diffEntitySetType", thisE.getName(), thatE.getType()));
                    diff.append("</li>");
                }
            }
        }
        
        // Report differencies between CERRelationshipSet properties
        for (Map.Entry<SimulatedElement, SimulatedElement> entry : elementMap.entrySet()) {
            if ((entry.getKey() instanceof CERRelationshipSet) && (entry.getValue() instanceof CERRelationshipSet)) {
                CERRelationshipSet thisE = (CERRelationshipSet) entry.getKey();
                CERRelationshipSet thatE = (CERRelationshipSet) entry.getValue();
                if (thisE.getType() != thatE.getType()) {
                    
                    diff.append("<li>");
                    diff.append(labels.getFormatted("diffRelationshipSetType", thisE.getName(),
                            labels.getFormatted("relationshipSetTypeName",thatE.getType())
                            ));
                    diff.append("</li>");
                }
            }
        }
        
        // Report differencies between CERISA properties
        for (Map.Entry<SimulatedElement, SimulatedElement> entry : elementMap.entrySet()) {
            if ((entry.getKey() instanceof CERISA) && (entry.getValue() instanceof CERISA)) {
                CERISA thisE = (CERISA) entry.getKey();
                CERISA thatE = (CERISA) entry.getValue();
                if (thisE.isDisjoint() != thatE.isDisjoint()) {
                    
                    diff.append("<li>");
                    diff.append(labels.getFormatted(
                            (thisE.isDisjoint()) ? "diffISANotDisjoint" : "diffISADisjoint"
                            ));
                    diff.append("</li>");
                }
            }
        }
        
        
        // Report differencies between CERAttribute properties
        for (Map.Entry<SimulatedElement, SimulatedElement> entry : elementMap.entrySet()) {
            if ((entry.getKey() instanceof CERAttribute) && (entry.getValue() instanceof CERAttribute)) {
                CERAttribute thisEA = (CERAttribute) entry.getKey();
                CERAttribute thatEA = (CERAttribute) entry.getValue();
                if (thatEA != null) {
                    if (thisEA.isPrimaryKey() != thatEA.isPrimaryKey()) {
                        
                        diff.append("<li>");
                        diff.append(labels.getFormatted(
                                (thisEA.isPrimaryKey()) ? "diffAttributeNotPrimaryKey" : "diffAttributePrimaryKey",
                                thisEA.getName())
                                );
                        diff.append("</li>");
                    }
                    if (thisEA.isDiscriminating() != thatEA.isDiscriminating()) {
                        
                        diff.append("<li>");
                        diff.append(labels.getFormatted(
                                (thisEA.isDiscriminating()) ? "diffAttributeNotDiscriminating" : "diffAttributeDiscriminating",
                                thisEA.getName())
                                );
                        diff.append("</li>");
                    }
                    if (thisEA.isMultivalued() != thatEA.isMultivalued()) {
                        
                        diff.append("<li>");
                        diff.append(labels.getFormatted(
                                (thisEA.isDiscriminating()) ? "diffAttributeNotMultivalued" : "diffAttributeMultivalued",
                                thisEA.getName())
                                );
                        diff.append("</li>");
                    }
                    if (thisEA.isDerived() != thatEA.isDerived()) {
                        
                        diff.append("<li>");
                        diff.append(labels.getFormatted(
                                (thisEA.isDerived()) ? "diffAttributeNotDerived" : "diffAttributeDerived",
                                thisEA.getName())
                                );
                        diff.append("</li>");
                    }
                }
            }
        }
        
        // Report differencies between connection properties
        for (Map.Entry<SimulatedRelationship, SimulatedRelationship> entry : connectionMap.entrySet()) {
            if ((entry.getKey() instanceof CERConnection) && (entry.getValue() instanceof CERConnection)) {
                CERConnection thisC = (CERConnection) entry.getKey();
                CERConnection thatC = (CERConnection) entry.getValue();
                
                if (thisC.getCardinality() != thatC.getCardinality()) {
                    
                    diff.append("<li>");
                    diff.append(labels.getFormatted("diffRelationshipCardinality", thisC.getStart().getName(), thisC.getEnd().getName(), thatC.getCardinality()));
                    diff.append("</li>");
                }
                
                String thisRL = thisC.getName();
                String thatRL = thatC.getName();
                boolean thatRLTraversable = thatC.isTraversable();
                if (thatRLTraversable && thatRL != null && (thisRL == null || ! thisRL.equals(thatRL))) {
                    diff.append("<li>");
                    diff.append(labels.getFormatted("diffRelationshipRoleName", thisC.getStart().getName(), thisC.getEnd().getName(), thatRL));
                    diff.append("</li>");
                }
                
                if (thisC.isTotal() != thatC.isTotal()) {
                    
                    diff.append("<li>");
                    diff.append(labels.getFormatted("diffRelationshipTotal", thisC.getStart().getName(), thisC.getEnd().getName(), thatC.isTotal() ? 1 : 0));
                    diff.append("</li>");
                }
            }
        }
        
        if (diff.length() == 0) {
            return null;
        } else {
            diff.insert(0, "<html><ul>");
            diff.append("</ul>");
            return diff.toString();
        }
    }
}